#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#ifndef UNITTEST
#include <sys/neutrino.h>
#include <devctl.h>
#include <sys/syspage.h>

#include "input/mtouch_log.h"
#include "input/mtouch_driver.h"
#include "input/event_types.h"

#include "input/inputtrace.h"
#include "input/parseopts.h"

#include "dcmd_pm.h"
#else
#include <fcntl.h>
#endif

#include "touch.h"
#include "common.h"

static int cypress_core_suspend(cypress_dev_t* cd)
{
    int mask_count;

    /*
     * This will not prevent resume
     * Required to prevent interrupts before i2c awake
    */
    mask_count = InterruptMask(cd->tp_intr, cd->tp_iid);
    mtouch_info(cd->log_name, "%s-maskCnt[%d]", __FUNCTION__,mask_count);


    /* stop the timer */
    if (EOK != cypress_set_heartbeat_timer(cd, 0)) {
        mtouch_error(cd->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
        return -1;
    }

    cypress_release_touch(cd);

    return 0;
}

static int cypress_core_resume(cypress_dev_t* cd)
{
    int mask_count;
    /* set driver state to unknown state */
    /* During suspend, a sleep signal is sent from VCU to FIDM MCU, MCU in the display will turn off
    the power to the touch IC. During Resume, a wake up signal is sent from VCU to MCU, then MCU
    power up the touch IC and reset the touch controller. It is required to re-intitialize the touch
    driver. so, setting the driver internal state to unknown will make driver to reinitialize touch
    HW IC */
    cd->driver_state = UNKNOWN;

    /* enable interrupts - After the resuming the thread and thread ready to serve the interrupt */
    mask_count = InterruptUnmask(cd->tp_intr, cd->tp_iid);
    mtouch_info(cd->log_name, "%s-maskCnt[%d]", __FUNCTION__,mask_count);

    return 0;
}

void *cypress_ext_msg_handler(void *arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    struct pm_ack_s ack = { .rc = 0 };
    fidm_touch_msg_u msg;
    int ret = -1, rc, getmodecy;
    int rcvid;
    struct _msg_info  info;
    int set_haptic_status = -1;

    /* get I/O privileges for GPIO */
    if (-1 == ThreadCtl(_NTO_TCTL_IO, 0)) {
        mtouch_error(cypress->log_name, "Failed to get IO privileges!");
    }

    while (1) {
        rcvid = MsgReceive(cypress->fidm_attach->chid, &msg, sizeof(msg), &info);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno) || (rcvid == -1)) {
                mtouch_error(cypress->log_name, "MsgReceive failed: %s", strerror(errno));
                error_memory("Cypress_Touch: MsgReceive failed: %s", strerror(errno));
                continue;
            }
        } else if (rcvid == 0) {
            /* Pulse received */
            switch (msg.pulse.code) {
                case CYPRESS_PM_PREPARE_PULSE_CODE:
                    mtouch_info(cypress->log_name, "Received CYPRESS_PM_PREPARE_PULSE");

                    if(cypress->cy_str_state == PM_STATE_PREPARE)
                    {
                        mtouch_info(cypress->log_name, "Ignoring the CYPRESS_PM_PREPARE_PULSE since mtouch driver is already in PREPARE state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    }
                    else
                    {
                        /* prepare cypress suspend */
                        // do nothing
                        if (ack.rc) {
                            mtouch_error(cypress->log_name, "Failed to handle CYPRESS_PM_SUSPEND_PULSE");
                            error_memory("Cypress_Touch: Failed to handle CYPRESS_PM_SUSPEND_PULSE");
                        } else {
                            /* Update CYPRESS PM state */
                            cypress->cy_pm_state = PM_STATE_PREPARE;
                            cypress->cy_str_state = PM_STATE_PREPARE;
                        }
                    }

                    /* send ACK to power manager */
                    ack.state = PM_STATE_PREPARE;
                    ret = devctl(cypress->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(cypress->log_name, "devctl(DCMD_PM_ACK) for CYPRESS_PM_PREPARE_PULSE failed: %d %s\n",
                                     errno, strerror(errno));
                        error_memory("Cypress_Touch: devctl(DCMD_PM_ACK) for CYPRESS_PM_PREPARE_PULSE failed: %d %s\n",
                                     errno, strerror(errno));
                    }
                    break;

                case CYPRESS_PM_SUSPEND_PULSE_CODE:
                    mtouch_info(cypress->log_name, "Received CYPRESS_PM_SUSPEND_PULSE");
					
					/* Send ACK to Power Manager */
                    ack.state = PM_STATE_SUSPEND;
					/* Send ACK.rc = 0 to Power Manager as susd will only be expecting ack.state only */
                    ack.rc = 0;
                    ret = devctl(cypress->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(cypress->log_name, "devctl(DCMD_PM_ACK) for CYPRESS_PM_SUSPEND_PULSE failed: %d %s\n",
                                     errno, strerror(errno));
                        error_memory("Cypress_Touch: devctl(DCMD_PM_ACK) for CYPRESS_PM_SUSPEND_PULSE failed: %d %s\n",
                                     errno, strerror(errno));
                    }

                    /*Check for suspend state variable. As pm state varaiable can be updated during controller initialization*/
                    if(cypress->cy_str_state == PM_STATE_SUSPEND)
                    {
                        mtouch_info(cypress->log_name, "Ignoring the CYPRESS_PM_SUSPEND_PULSE since mtouch driver is already in SUSPEND state");
			ack.rc = 0;
                    }
                    else
                    {
                        /* process cypress suspend */
                        ack.rc = cypress_core_suspend(cypress);
                        if (ack.rc) {
                            mtouch_error(cypress->log_name, "Failed to handle CYPRESS_PM_SUSPEND_PULSE");
                            error_memory("Cypress_Touch: Failed to handle CYPRESS_PM_SUSPEND_PULSE");
                        } else {
                            /* Update CYPRESS PM state */
                            cypress->cy_pm_state = PM_STATE_SUSPEND;
                            cypress->cy_str_state = PM_STATE_SUSPEND;
                        }
                    }
                    break;

                case CYPRESS_PM_RESUME_PULSE_CODE:
                    mtouch_info(cypress->log_name, "Received CYPRESS_PM_RESUME_PULSE");
                    
                    /* Send ACK to Power Manager first to unblock power manager from sending pulse to other clients*/
                     /*Send ACK.rc = 0 to Power Manager as susd will only be expecting ack.state only*/
                    ack.state = PM_STATE_RESUME;
                    ack.rc = 0;
                    ret = devctl(cypress->pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(cypress->log_name, "devctl(DCMD_PM_ACK) for CYPRESS_PM_RESUME_PULSE: failed: %d, %s\n",
                        errno, strerror(errno));
                        error_memory("Cypress_Touch: devctl(DCMD_PM_ACK) for CYPRESS_PM_RESUME_PULSE: failed: %d, %s\n",
                        errno, strerror(errno));
                    } else {
                        mtouch_info(cypress->log_name, "Sent Ack to powermanager successfully, ack.state:%d", ack.state);
                    }
                    
                    if(cypress->cy_str_state == PM_STATE_RESUME)
                    {
                        mtouch_info(cypress->log_name, "Ignoring the CYPRESS_PM_RESUME_PULSE since mtouch driver is already in RESUME state");
                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    }
                    else
                    {
                       getmodecy = cypress_get_mode(cypress);
                       if(OPERATING_MODE != getmodecy)
                       {
                            mtouch_error(cypress->log_name,"Cntrler not in Operating mode[%d],Resetting it, to out from CAT mode",getmodecy);
                            cypress_controller_reset(cypress);
                       }
                        /* process cypress resume */
                        ack.rc = cypress_core_resume(cypress);
                        if (ack.rc) {
                            mtouch_error(cypress->log_name, "Failed to handle CYPRESS_PM_RESUME_PULSE");
                            error_memory("Cypress_Touch: Failed to handle CYPRESS_PM_RESUME_PULSE");
                        } else {
                            /* Update CYPRESS PM state */
                           cypress->cy_pm_state = PM_STATE_RESUME;
                           cypress->cy_str_state = PM_STATE_RESUME;
                        }
                    }
                    break;

                case _PULSE_CODE_DISCONNECT:
                       mtouch_info(cypress->log_name, "kernel closed connection %d, detaching it",  info.coid);
                       ConnectDetach(info.scoid);
                       break;

                default:
                    mtouch_warn(cypress->log_name, "Received unknown pulse from msgthread: %d,pid = %d, coid = %d, msglen=%d, ",
                                 msg.pulse.code, info.pid, info.coid, info.msglen);
                    break;
            }
        }
        else {
            /* if cypress is in suspend/prepare state, return EAGAIN to try after sometime */
            if (((cypress->cy_pm_state != PM_STATE_RESUME) || (cypress->cy_str_state == PM_STATE_SUSPEND)) && (msg.type != FIDM_TOUCH_GET_GROUPID)) {
                MsgError(rcvid, -EAGAIN);
                continue;
            }

            /* if cypress is not initialized or not in operating mode, dont process messages */
            /* FIDM_TOUCH_GET_GROUPID can be accepted even if the the driver and the controller state is not in operating*/
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
            if ((cypress->driver_state != OPERATING) && (msg.type != FIDM_TOUCH_GET_GROUPID) && (cypress->sysinfo_to_operating_mode_state != false))
#else
             if ((cypress->driver_state != OPERATING) && (msg.type != FIDM_TOUCH_GET_GROUPID))
#endif
            {
                MsgError(rcvid, -EAGAIN);
                continue;
            }

            /* msg received */
            switch(msg.type) {
                case FIDM_TOUCH_GET_RTD_DATA:
                    if (cypress->verbose > 5)
                        mtouch_info(cypress->log_name, "Received FIDM_TD_MSG_GET_RTD_DATA");

                    if (cypress->rtd_enable == 1)  {
                        if (cypress->rtd_readflag == 1) {
                            ret = MsgReply(rcvid, 0, cypress->rtd_data, cypress->rtd_len);
                            if (ret == EOK) {
                                cypress->rtd_readflag = 0;
                            }
                            else {
                                MsgError(rcvid, -errno);
                            }
                            if (cypress->verbose > 5) {
                                mtouch_info(cypress->log_name, "Msg replied with rtd data: %d", errno);
                            }
                        }
                        else {
                            MsgError(rcvid, -EAGAIN);
                        }
                    }
                    else {
                        MsgError(rcvid, -ENOTSUP);
                    }
                    break;

                case FIDM_TOUCH_DISABLE_INTERRUPT:
                    if (cypress->verbose > 5)
                        mtouch_info(cypress->log_name, "Received FIDM_TOUCH_DISABLE_INTERRUPT");

                    InterruptMask(cypress->tp_intr, cypress->tp_iid);
                    MsgReply(rcvid, 0, NULL, 0);
                    break;

                case FIDM_TOUCH_ENABLE_INTERRUPT:
                    if (cypress->verbose > 5)
                        mtouch_info(cypress->log_name, "Received FIDM_TOUCH_ENABLE_INTERRUPT");

                    InterruptUnmask(cypress->tp_intr, cypress->tp_iid);
                    MsgReply(rcvid, 0, NULL, 0);
                    break;

                case FIDM_TOUCH_GET_GROUPID:
                      mtouch_info(cypress->log_name, "Received FIDM_TOUCH_GET_GROUPID");
                     if(cypress->display_group_id < 0) {
                         ret = cypress_get_display_grpid(cypress);
                         if(ret != 0)
                         {
                            mtouch_error(cypress->log_name,"failed to fetch group id: %d again", cypress->display_group_id);
                            MsgError(rcvid, -errno);
                         } else {
                            ret = MsgReply(rcvid, 0, &cypress->display_group_id, sizeof(cypress->display_group_id));
                            if (ret != EOK) {
                                 MsgError(rcvid, -errno);
                            }
                         }
                     } else {
                         ret = MsgReply(rcvid, 0, &cypress->display_group_id, sizeof(cypress->display_group_id));
                         mtouch_info(cypress->log_name, "GROUPID %d ", cypress->display_group_id);
                         if(ret != EOK) {
                            MsgError(rcvid, -errno);
                         }
                     }
                     break;

                case FIDM_TOUCH_SET_HAPTIC_DATA:
                    if(cypress_get_mode(cypress) == OPERATING_MODE)
                    {
                        mtouch_info(cypress->log_name, "Received FIDM_TOUCH_SET_HAPTIC_DATA, data_len %d", msg.haptic_data.len);
                        cypress->haptic_flag = true;
                        rc = cypress_set_haptic_data(cypress, msg.haptic_data.buffer, msg.haptic_data.len);
                        set_haptic_status = rc;
                        ret = MsgReply(rcvid, rc, NULL, 0);
                        if (ret == EOK) {
                         mtouch_info(cypress->log_name, "Msg replied with set haptic data result: %d", rc);
                        }
                        else {
                          MsgError(rcvid, -errno);
                        }
                    }
                    else
                    {
                        set_haptic_status = -1;
                        mtouch_error(cypress->log_name, "SET_HAPTIC Request failed: Controller is not in Operating mode");
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;

                case FIDM_TOUCH_PLAYHAPTIC:
                    //if((cypress_get_mode(cypress) == OPERATING_MODE) && (set_haptic_status == EOK))
                    if (set_haptic_status == EOK)
                    {
                        mtouch_info(cypress->log_name, "Received FIDM_TOUCH_PLAYHAPTIC");
                        if (cypress->play_haptic_config == true) {
                            rc = cypress_play_haptic_data(cypress, msg.haptic_data.buffer);
                            mtouch_info(cypress->log_name, "Play haptic is enabled");
                        }
                        else {
                            rc = 0;
                            mtouch_info(cypress->log_name, "Play haptic is disabled");
                            /* commenting play haptic as Infineon confirmed that set haptic operation would suffice */
                            //rc = cypress_play_haptic_data(cypress, msg.haptic_data.buffer);
                        }
                        ret = MsgReply(rcvid, rc, NULL, 0);
                        if (ret == EOK) {
                            mtouch_info(cypress->log_name, "Play haptic always success");
                        }
                        else {
                            mtouch_error(cypress->log_name, "MsgReply failed with error - %d",errno);
                            MsgError(rcvid, -errno);
                        }
                    }
                    else
                    {
                        mtouch_error(cypress->log_name, "PLAY_HAPTIC Request failed: SET_HAPTIC_STATUS - %d; Incorrect mode",set_haptic_status);
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;

                default:
                    mtouch_warn(cypress->log_name, "Received unknown msg from msgthread: %d, pid = %d, coid = %d, msglen=%d, ",
                                 msg.type, info.pid, info.coid, info.msglen);
                    MsgError(rcvid, -EINVAL);
                    break;
            }
        }
    }

    return NULL;
}


int cypress_register_pm(cypress_dev_t* cypress)
{
    struct pm_register_s pm_reg;
    int ret;

    cypress->pm_fd = open(cypress->pm_dev, O_RDWR  | O_CLOEXEC);
    if (cypress->pm_fd  == -1) {
        mtouch_error(cypress->log_name, "dev pm open() failed %d", strerror(errno));
        goto fail;
    }

    memset(&pm_reg, 0x0, sizeof(struct pm_register_s));
    INIT_PM_REGISTER_STRUCT(&pm_reg);

    strlcpy(pm_reg.name, cypress->log_name, sizeof(pm_reg.name));
    pm_reg.pulse_codes[PM_STATE_PREPARE] = CYPRESS_PM_PREPARE_PULSE_CODE;
    pm_reg.pulse_codes[PM_STATE_SUSPEND] = CYPRESS_PM_SUSPEND_PULSE_CODE;
    pm_reg.pulse_codes[PM_STATE_RESUME] = CYPRESS_PM_RESUME_PULSE_CODE;
    pm_reg.pulse_codes[PM_STATE_COMPLETE] = -1;
    pm_reg.priority = PM_PRIO_LEVEL_0;
    pm_reg.flags = 0;
    pm_reg.chid = cypress->fidm_attach->chid;

    /* Register Cypresss with Power Manager */
    ret = devctl(cypress->pm_fd, DCMD_PM_REGISTER, &pm_reg, sizeof(struct pm_register_s), NULL);
    if (ret != EOK) {
        mtouch_error(cypress->log_name, "devctl() failed: %s\n", strerror(errno));
        error_memory("Cypress_Touch: devctl() failed: %s\n", strerror(errno));
        goto fail_pm;
    }
    mtouch_info(cypress->log_name, "/dev/pm registration successful %d\n", ret);

    /* Success */
    return 0;

fail_pm:
    close(cypress->pm_fd);
fail:
    return -1;
}
